<?php
/**
 * Admin area settings and hooks.
 *
 * @package DR_PRO
 * @subpackage  DR_PRO
 */

namespace DR_PRO;

defined( 'ABSPATH' ) || exit;

/**
 * Global Settings.
 */
class Delicious_Recipes_Pro_Public {

	/**
	 * Constructor.
	 */
	public function __construct() {

		$this->init();
	}

	/**
	 * Initialization.
	 *
	 * @since 1.0.0
	 * @access private
	 *
	 * @return void
	 */
	private function init() {

		// Initialize hooks.
		$this->init_hooks();
		$this->includes();

		// Allow 3rd party to remove hooks.
		do_action( 'wp_delicious_pro_public_unhook', $this );
	}

	/**
	 * Initialize hooks.
	 *
	 * @since 1.0.0
	 * @access private
	 *
	 * @return void
	 */
	private function init_hooks() {

		// Action to enqueue scripts added before license status validity to make sure css scripts runs/loads.
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );

		// Check for license status.
		if ( ! delicious_recipe_pro_check_license_status() ) {
			return;
		}

		add_action( 'init', array( 'Delicious_Recipes_Pro_Shortcodes', 'init' ), 99999999 );

		// Public Scripts.
		add_action( 'wp_enqueue_scripts', array( $this, 'set_translation_scripts' ), 99999999999 );
		add_filter( 'delicious_recipes_user_dashboard_menus', array( $this, 'filter_ud_menus' ) );
		add_filter( 'delicious_recipes_user_dashboard_svgs', array( $this, 'ud_menu_svgs' ) );

		// Add dynamic CSS.
		add_action( 'wp_head', array( $this, 'load_dynamic_css' ), 99 );

		// Hooks for sending emails.
		add_action( 'delicious_recipes_frontend_recipe_submitted', array( $this, 'new_recipe_submission' ), 20, 3 );
		add_action( 'transition_post_status', array( $this, 'track_recipe_published' ), 10, 3 );

		// Adds the floating bar to single recipe.
		add_action( 'wp_footer', array( $this, 'get_floating_bar' ) );

		// Remove comments hooks and filter.
		add_action(
			'wp_delicious_public_unhook',
			function ( $instance ) {
				remove_action( 'comment_form_logged_in_after', array( $instance, 'dr_comment_form_rating_fields' ), 10, 1 );
				remove_action( 'comment_form_after_fields', array( $instance, 'dr_comment_form_rating_fields' ), 10, 1 );
				remove_action( 'comment_post', array( $instance, 'dr_save_comment_rating' ), 10, 1 );
				remove_filter( 'comment_text', array( $instance, 'dr_add_comment_review_after_text' ) );
			}
		);

		// Add the related posts content on recipe page.
		add_filter( 'the_content', array( $this, 'insert_related_recipes_content' ) );

		// Modify the dashboard page if dashboard portal is enabled.
		add_action( 'init', array( $this, 'modify_dashboard_page' ) );

		// Filter currently used themes header.php to fit WP Delicious recipe banner.
		add_filter( 'template_include', array( $this, 'load_edited_header' ) );
	}

	/**
	 * Modify the dashboard page if dashboard portal is enabled.
	 *
	 * @return void
	 */
	public function modify_dashboard_page() {
		$global_settings  = delicious_recipes_get_global_settings();
		$dashboard_portal = isset( $global_settings['dashboardPortal'][0] ) ?? 'no';
		$dashboard_page   = isset( $global_settings['dashboardPage'] ) ? $global_settings['dashboardPage'] : '';

		if ( $dashboard_portal && $dashboard_page ) {
			// check if the current page has the shortcode 'dr_user_dahsboard'.
			if ( has_shortcode( get_the_content(), 'dr_user_dashboard' ) ) {
				show_admin_bar( false );
			}
			// exclude header.php and footer.php from the dashboard page.
			add_filter(
				'template_include',
				function ( $template ) {
					$global_settings = delicious_recipes_get_global_settings();
					$dashboard_page  = $global_settings['dashboardPage'];
					if ( is_page( $dashboard_page ) && is_user_logged_in() ) {
						// Custom template that excludes header/footer but includes scripts and styles.
						$template = plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'templates/dashboard-portal.php';
					}
					return $template;
				}
			);
		}
	}

	/**
	 * Load edited header.
	 *
	 * @param string $template Template.
	 * @return string
	 *
	 * @since 2.2.3
	 */
	public function load_edited_header( $template ) {
		if ( ! is_recipe() ) {
			return $template;
		}

		if ( is_recipe() && is_singular() ) {
			$global_settings        = delicious_recipes_get_global_settings();
			$selected_banner_layout = array();
			$banner_layout_id       = 'default';
			$theme                  = wp_get_theme();
			$theme_name             = $theme->get( 'Name' );

			if ( function_exists( 'DEL_RECIPE_PRO' ) ) {
				$license_validity_bool = delicious_recipe_pro_check_license_status();
				if ( $license_validity_bool ) {
					$selected_banner_layout = isset( $global_settings['selectedBannerLayout'] ) ? $global_settings['selectedBannerLayout'] : array();
					$banner_layout_id       = isset( $selected_banner_layout['id'] ) ? $selected_banner_layout['id'] : 'default';
				}
			}

			if ( isset( $selected_banner_layout ) && 'default' !== $banner_layout_id && array( 'yes' ) !== $global_settings['enableElementorSupport'] ) {
				if ( $theme_name === 'Foxiz' ) {
					$this->modify_foxiz_theme_header();
				} else {
					$this->modify_theme_header();
				}
			}
		}
		return $template;
	}

	/**
	 * Modify theme header.
	 *
	 * @since 2.2.3
	 */
	public function modify_theme_header() {
		// Get the path to the current theme's header.php file.
		$theme_header_file = get_template_directory() . '/header.php';

		// Check if the header.php file exists.
		if ( ! file_exists( $theme_header_file ) ) {
			return;
		}

		// Start output buffering.
		ob_start();

		// Include the original header.php content.
		include $theme_header_file;

		// Get the buffered content.
		$header_content = ob_get_clean();

		// Append the custom div to the header content.
		while ( have_posts() ) {
			the_post();
			$custom_div = delicious_recipes_get_template_html( 'recipe/header.php' );
		}

		$modified_header_content = str_replace( '</header>', '</header>' . $custom_div, $header_content );

		ob_clean();

		// Output the modified header content.
		echo $modified_header_content;
	}

	/**
	 * Modify theme header when foxiz theme is activated
	 */
	public function modify_foxiz_theme_header() {

		ob_start();
		get_header();  // Get the header content.
		$full_page_content = ob_get_clean();
		while ( have_posts() ) {
			the_post();
			$custom_div = delicious_recipes_get_template_html( 'recipe/header.php' );
		}
		echo $full_page_content . $custom_div;
	}


	/**
	 * Set translation scripts.
	 *
	 * @return void
	 */
	public function set_translation_scripts() {
		wp_set_script_translations( 'dr-pro-usr-dashboard', 'delicious-recipes-pro' );
		wp_set_script_translations( 'delicious-recipes-pro', 'delicious-recipes-pro' );
	}

	/**
	 * Includes
	 *
	 * @return void
	 */
	private function includes() {

		// Check for license status.
		if ( ! delicious_recipe_pro_check_license_status() ) {
			return;
		}

		if ( $this->is_request( 'frontend' ) ) {
			require plugin_dir_path( __FILE__ ) . '/classes/class-delicious-recipes-pro-shortcodes.php';
			require plugin_dir_path( __FILE__ ) . '/classes/class-delicious-recipes-pro-template-hooks.php';
			require plugin_dir_path( __FILE__ ) . '/classes/class-delicious-recipes-pro-emojis.php';
		}
	}

	/**
	 * Enqueue Public Scripts
	 *
	 * @return void
	 */
	public function enqueue_scripts() {

		$global_settings = delicious_recipes_get_global_settings();

		if (
			! (
				is_recipe()
				|| is_recipe_shortcode()
				|| is_recipe_search()
				|| is_recipe_taxonomy()
				|| is_recipe_block()
				|| is_recipe_dashboard()
				|| is_page_template( 'templates/pages/recipe-badges.php' )
				|| is_page_template( 'templates/pages/recipe-cooking-methods.php' )
				|| is_page_template( 'templates/pages/recipe-courses.php' )
				|| is_page_template( 'templates/pages/recipe-cuisines.php' )
				|| is_page_template( 'templates/pages/recipe-dietary.php' )
				|| is_page_template( 'templates/pages/recipe-keys.php' )
				|| is_page_template( 'templates/pages/recipe-tags.php' )
				|| has_shortcode( get_the_content(), 'dr_guest_recipe_submission' )
				|| has_shortcode( get_the_content(), 'dr_submit_recipe' )
				|| has_shortcode( get_the_content(), 'dr_my_recipes' )
				|| has_shortcode( get_the_content(), 'dr_favorites_recipes' )
				|| has_shortcode( get_the_content(), 'dr_login' )
				|| has_shortcode( get_the_content(), 'dr_recipe_equipments' )
			)
		) {
			return;
		}

		if ( is_recipe() && (isset( $global_settings['relatedRecipes'] ) && $global_settings['relatedRecipes'] || isset( $global_settings['enableEquipments'] ) && $global_settings['enableEquipments']) ) {
			wp_enqueue_style( 'delicious-recipe-splide-css' );
		}

		wp_enqueue_style(
			'delicious-recipes-pro-new',
			plugin_dir_url( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/wpd-pro-global.css',
			array(),
			DELICIOUS_RECIPES_PRO_VERSION,
			'all'
		);

		if ( is_recipe() && $global_settings['enableDidYouMake'] && $global_settings['enableReviewImages'] ) {
			wp_enqueue_style( 'dropzone', plugin_dir_url( DELICIOUS_RECIPES_PLUGIN_FILE ) . 'assets/lib/dropzone/dropzone.min.css', array(), '5.9.2', 'all' );
			wp_enqueue_script( 'dropzone', plugin_dir_url( DELICIOUS_RECIPES_PLUGIN_FILE ) . 'assets/lib/dropzone/dropzone.min.js', array(), '5.9.2', true );
			wp_add_inline_script( 'dropzone', 'Dropzone.autoDiscover = false;' );
		}

		$pro_ud_deps = include_once plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/usrdashboard.asset.php';

		if ( is_recipe_dashboard()
			|| has_shortcode( get_the_content(), 'dr_guest_recipe_submission' )
			|| has_shortcode( get_the_content(), 'dr_submit_recipe' )
			|| has_shortcode( get_the_content(), 'dr_my_recipes' )
			|| has_shortcode( get_the_content(), 'dr_favorites_recipes' )
			|| has_shortcode( get_the_content(), 'dr_login' )
		) {
			// Recipe usrdashboard screen assets.
			wp_register_script(
				'dr-pro-usr-dashboard',
				plugin_dir_url( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/usrdashboard.js',
				$pro_ud_deps['dependencies'],
				filemtime( plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/usrdashboard.js' ),
				true
			);
			$max_upload_size = delicious_recipes_get_max_upload_size();
			// Add localization vars.
			wp_localize_script(
				'dr-pro-usr-dashboard',
				'DeliciousRecipesProUD',
				array(
					'siteURL'                   => esc_url( home_url( '/' ) ),
					'pluginUrl'                 => esc_url( plugin_dir_url( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) ),
					'ajaxURL'                   => admin_url( 'admin-ajax.php' ),
					'featuredImageNonce'        => wp_create_nonce( 'delicious-recipes-featured-image-nonce' ),
					'instructionsImageNonce'    => wp_create_nonce( 'delicious-recipes-instructions-image-nonce' ),
					'galleryImagesNonce'        => wp_create_nonce( 'delicious-recipes-gallery-images-nonce' ),
					'maxUploadSize'             => esc_html( $max_upload_size ),
					'globalSettings'            => $global_settings,
					'maxUploadSizeBytes'        => wp_max_upload_size(),
					'recipeSubmissionStructure' => get_option( 'recipe_submission_structure' ),
				)
			);
			wp_enqueue_script( 'dr-pro-usr-dashboard' );
			wp_enqueue_style(
				'dr-pro-usr-dashboard',
				plugin_dir_url( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/dashboardPortal.css',
				array(),
				DELICIOUS_RECIPES_PRO_VERSION,
				'all'
			);
		}

		if ( isset( $global_settings['autoloadRecipes'] ) && array( 'yes' ) === $global_settings['autoloadRecipes'] && is_recipe() ) {
			wp_enqueue_script( 'infinite-scroll', plugin_dir_url( DELICIOUS_RECIPES_PLUGIN_FILE ) . 'assets/lib/infinite-scroll/infinite-scroll.pkgd.min.js', array( 'jquery' ), '4.0.1', true );
		}

		$dashboard_page   = isset( $global_settings['dashboardPage'] ) ? $global_settings['dashboardPage'] : '';
		$dashboard_portal = isset( $global_settings['dashboardPortal'][0] ) ?? 'no';
		if ( is_page( $dashboard_page ) && $dashboard_portal ) {
			wp_enqueue_script( 'delicious-recipes-pro', plugin_dir_url( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/wpdPublic.js', array(), filemtime( plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/wpdPublic.js' ), true );
		}

		wp_register_script( 'delicious-recipes-pro-cook-mode-js', plugin_dir_url( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/cookMode.js', array( 'jquery', 'wp-util' ), filemtime( plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/cookMode.js' ), true );
		wp_register_script( 'delicious-recipes-pro-equipment-js', plugin_dir_url( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/equipmentBlock.js', array(), filemtime( plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/equipmentBlock.js' ), true );

		if ( has_block( 'delicious-recipes/recipe-card' ) ) {
			wp_enqueue_script( 'delicious-recipes-pro-cook-mode-js' );
			wp_enqueue_script( 'delicious-recipes-pro-equipment-js' );
			wp_enqueue_style( 'delicious-recipe-splide-css' );
		}

		if ( ! is_admin() && is_recipe() ) {
			wp_enqueue_script(
				'delicious-recipes-pro',
				plugin_dir_url( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/wpdPublic.js',
				array( 'jquery', 'wp-util', 'wp-api-fetch' ),
				filemtime( plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/wpdPublic.js' ),
				true,
			);
				wp_enqueue_script( 'delicious-recipes-pro-equipment-js' );
				wp_enqueue_script( 'delicious-recipes-pro-cook-mode-js' );
		}

		if ( has_shortcode( get_the_content(), 'dr_recipe_equipments' ) ) {
			wp_enqueue_script( 'delicious-recipes-pro-equipment', plugin_dir_url( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/equipmentPublic.js', array(), filemtime( plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/equipmentPublic.js' ), true );
			wp_enqueue_style( 'delicious-recipes-pro-equipment', plugin_dir_url( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/equipmentPublic.css', array(), filemtime( plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/equipmentPublic.css' ), 'all' );
		}

		$enable_autoload  = isset( $global_settings['autoloadRecipes']['0'] ) && 'yes' === $global_settings['autoloadRecipes']['0'] ? true : false;
		$autoload_perpage = isset( $global_settings['autoloadPerPage'] ) && $global_settings['autoloadPerPage'] ? absint( $global_settings['autoloadPerPage'] ) : 4;
		$autoload_filter  = isset( $global_settings['autoloadFilter'] ) && $global_settings['autoloadFilter'] ? esc_attr( $global_settings['autoloadFilter'] ) : 'prev';
		$enable_reaction  = isset( $global_settings['emojiReactions']['0'] ) && 'yes' === $global_settings['emojiReactions']['0'] ? true : false;

		$delicious_recipes_pro = array(
			'ajax_url'              => admin_url( 'admin-ajax.php' ),
			'upload_review_pic_msg' => __( 'Click here or Drop new images to upload your recipe review pictures', 'delicious-recipes-pro' ),
			'enable_autoload'       => $enable_autoload,
			'autoload_perpage'      => $autoload_perpage,
			'autoload_filter'       => $autoload_filter,
			'autoload_selector'     => apply_filters( 'wp_delicious_autoload_selector', '.wpdelicious-outer-wrapper .content-area' ),
			'autoload_append'       => apply_filters( 'wp_delicious_autoload_append_selector', '.dr-single-recipe' ),
			'enable_reaction'       => $enable_reaction,
		);

		wp_localize_script( 'delicious-recipes-pro', 'delicious_recipes_pro', $delicious_recipes_pro );
	}

	/**
	 * What type of request is this?
	 *
	 * @param  string $type admin, ajax, cron or frontend.
	 * @return bool
	 */
	private function is_request( $type ) {
		switch ( $type ) {
			case 'admin':
				return is_admin();
			case 'ajax':
				return defined( 'DOING_AJAX' );
			case 'cron':
				return defined( 'DOING_CRON' );
			case 'frontend':
				return ( ! is_admin() || defined( 'DOING_AJAX' ) ) && ! defined( 'DOING_CRON' );
		}
	}

	/**
	 * Get default global settings values.
	 *
	 * @param array $dashboard_tabs Dashboard tabs.
	 */
	public function filter_ud_menus( $dashboard_tabs = array() ) {

		$global_settings          = delicious_recipes_get_global_settings();
		$enable_recipe_submission = isset( $global_settings['enableRecipeSubmission']['0'] ) && 'yes' === $global_settings['enableRecipeSubmission']['0'] ? true : false;

		if ( $enable_recipe_submission ) {
			$user_dashboard_tabs = apply_filters(
				'delicious_recipes_pro_user_dashboard_tabs',
				array(
					'recipe-submission' => array(
						'menu_title'      => __( 'Submit Recipe', 'delicious-recipes-pro' ),
						'menu_class'      => 'dr-ud_tab recipe-submission',
						'menu_content_cb' => array( $this, 'recipe_submission' ),
						'priority'        => 8,
						'svg'             => delicious_recipes_get_svg( 'recipe-submission', 'dashboard', '#000000' ),
					),
					'my-recipes'        => array(
						'menu_title'      => __( 'My Recipes', 'delicious-recipes-pro' ),
						'menu_class'      => 'dr-ud_tab my-recipes',
						'menu_content_cb' => array( $this, 'user_recipes' ),
						'priority'        => 9,
						'svg'             => delicious_recipes_get_svg( 'my-recipes', 'dashboard', '#000000' ),
					),
				)
			);

			$dashboard_tabs = array_merge( $dashboard_tabs, $user_dashboard_tabs );
		}
		return $dashboard_tabs;
	}

	/**
	 * UD svgs.
	 *
	 * @param array $dashboard_svgs Dashboard svgs.
	 */
	public function ud_menu_svgs( $dashboard_svgs = array() ) {

		$user_dashboard_svgs = array(
			'recipe-submission' => '<svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" clip-rule="evenodd" d="M9.40961 2.74286C9.73569 2.41677 10.2644 2.41677 10.5905 2.74286L15.5905 7.74286C15.9166 8.06894 15.9166 8.59764 15.5905 8.92373C15.2644 9.24981 14.7357 9.24981 14.4096 8.92373L10.835 5.34916V16.6666C10.835 17.1278 10.4612 17.5016 10 17.5016C9.53888 17.5016 9.16504 17.1278 9.16504 16.6666V5.34916L5.59047 8.92373C5.26439 9.24981 4.73569 9.24981 4.4096 8.92373C4.08352 8.59764 4.08352 8.06894 4.4096 7.74286L9.40961 2.74286Z" fill="#566267"/></svg>',

			'my-recipes'        => '<svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg"><g><path d="M13.4394 4.26394C12.8156 2.90298 11.4836 2 9.99986 2C8.5161 2 7.18414 2.90298 6.56166 4.26394C4.1973 3.76707 2 5.66523 2 8.16093C2 9.64573 2.77156 10.9721 3.99999 11.6591V17.3044C3.99999 17.6889 4.29812 18 4.66664 18H15.3334C15.7019 18 16 17.6889 16 17.3044V11.6602C17.2284 10.9718 18 9.64687 18 8.16204C18 5.66479 15.804 3.76659 13.4398 4.26357L13.4394 4.26394ZM15.0902 10.5677C14.8349 10.6714 14.6665 10.9292 14.6665 11.2158V13.1298H5.33329V11.2158C5.33329 10.9292 5.16491 10.6728 4.90957 10.5677C3.95169 10.1774 3.33334 9.23266 3.33334 8.16119C3.33334 6.73685 4.44443 5.57752 5.8095 5.57752C6.12281 5.57752 6.4306 5.63801 6.72183 5.75899C7.09174 5.91164 7.5072 5.69562 7.6107 5.29526C7.90331 4.17479 8.88605 3.39134 9.99989 3.39134C11.1137 3.39134 12.0965 4.17479 12.3891 5.29526C12.494 5.69561 12.9095 5.91166 13.278 5.75899C13.5692 5.63801 13.877 5.57752 14.1903 5.57752C15.5554 5.57752 16.6665 6.73685 16.6665 8.16119C16.6665 9.23266 16.0481 10.1774 15.0902 10.5677Z" fill="#566267"/></g></svg>',
		);

		$dashboard_svgs = array_merge( $dashboard_svgs, $user_dashboard_svgs );

		return $dashboard_svgs;
	}

	/**
	 * Recipe submission form UI
	 *
	 * @return void
	 */
	public function recipe_submission() {
		echo '<div id="dr-pro-submit-recipe" data-rest-nonce="' . wp_create_nonce( 'wp_rest' ) . '" data-guest-mode="' . false . '"></div>';
	}

	/**
	 * Recipe Listing in User Dahboard.
	 *
	 * @return void
	 */
	public function user_recipes() {
		echo '<div id="dr-pro-urlist" data-rest-nonce="' . wp_create_nonce( 'wp_rest' ) . '"></div>';
	}

	/**
	 * Load Dynamic CSS values.
	 *
	 * @return void
	 */
	public function load_dynamic_css() {

		if ( is_recipe_pro_shortcode() ) {
			delicious_recipes_get_template( 'global/dynamic-css.php' );
		}
	}

	/**
	 * New Recipe submission notification email.
	 *
	 * @param int   $customer_id  Customer ID.
	 * @param array $recipe_id    Recipe ID.
	 * @param bool  $disable_email Disable email.
	 */
	public function new_recipe_submission( $customer_id, $recipe_id, $disable_email = false ) {

		$global_settings     = delicious_recipes_get_global_settings();
		$enable_notification = isset( $global_settings['enableNotification']['0'] ) && 'yes' === $global_settings['enableNotification']['0'] ? true : false;

		if ( ! $customer_id || ! $enable_notification || $disable_email ) {
			return;
		}

		// Email Notification to Admin.
		include plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . '/includes/classes/emails/admin-new-recipe-submission.php';
		$email = new \Admin_New_Recipe_Submission();
		$email->trigger( $customer_id, $recipe_id );

		$enable_pending_mode = isset( $global_settings['enablePendingMode']['0'] ) && 'yes' === $global_settings['enablePendingMode']['0'] ? true : false;

		// Email Notification to Customer.
		if ( $enable_pending_mode ) {
			include plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . '/includes/classes/emails/new-recipe-submission.php';
			$email = new \New_Recipe_Submission();
			$email->trigger( $customer_id, $recipe_id );
		} else {
			include plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . '/includes/classes/emails/recipe-published.php';
			$email = new \Recipe_Published();
			$email->trigger( $customer_id, $recipe_id );
		}
	}

	/**
	 * Send a recipe published email when a recipe is published.
	 *
	 * @param string $new_status New post_status.
	 * @param string $old_status Previous post_status.
	 * @param object $post WordPress post.
	 */
	public function track_recipe_published( $new_status, $old_status, $post ) {
		$global_settings     = delicious_recipes_get_global_settings();
		$enable_notification = isset( $global_settings['enableNotification']['0'] ) && 'yes' === $global_settings['enableNotification']['0'] ? true : false;

		if (
			DELICIOUS_RECIPE_POST_TYPE !== $post->post_type ||
			'publish' !== $new_status ||
			'publish' === $old_status ||
			! $enable_notification ||
			! user_can( $post->post_author, 'edit_delicious_recipes' )
		) {
			return;
		}

		// Email Notification to Customer.
		include plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . '/includes/classes/emails/recipe-published.php';
		$email = new \Recipe_Published();
		$email->trigger( $post->post_author, $post->ID );
	}

	/**
	 * Adds the floating bar to single recipe
	 */
	public function get_floating_bar() {

		$global_settings     = delicious_recipes_get_global_settings();
		$enable_floating_bar = isset( $global_settings['enableFloatingBar']['0'] ) && 'yes' === $global_settings['enableFloatingBar']['0'] ? true : false;

		if ( ! is_recipe() || ! $enable_floating_bar ) {
			return;
		}

		if ( isset( $_GET['print_recipe'] ) && 'true' == $_GET['print_recipe'] ) {
			return;
		}

		ob_start();

			delicious_recipes_pro_get_template( 'recipe/floating-bar.php' );

			delicious_recipes_pro_get_template( 'recipe/exit-reading-mode.php' );

		echo ob_get_clean();
	}

	/**
	 * Insert Related Recipes Posts Content into the single recipe page.
	 *
	 * @param string $content Post content.
	 */
	public function insert_related_recipes_content( $content ) {
		if ( ! is_recipe() ) {
			return $content;
		}

		$global_settings        = delicious_recipes_get_global_settings();
		$enable_related_recipes = isset( $global_settings['relatedRecipes']['0'] ) && 'yes' === $global_settings['relatedRecipes']['0'] ? true : false;
		if ( ! $enable_related_recipes ) {
			return $content;
		}

		$show_after      = isset( $global_settings['showAfterParagraph'] ) ? absint( $global_settings['showAfterParagraph'] ) : 3;
		$paragraph_count = substr_count( $content, '</p>' );
		$paragraph_id    = $paragraph_count < $show_after ? -1 : $show_after;

		if ( ( $paragraph_count < $show_after ) || 0 === $show_after ) {
			$content = delicious_recipes_pro_get_related_recipes_content() . $content;
		} else {
			$content = delicious_recipes_pro_insert_after_paragraph( delicious_recipes_pro_get_related_recipes_content(), $paragraph_id, $content );
		}

		return $content;
	}
}
