<?php
/**
 * REST API: Delicious_Recipes_Pro_REST_Ingredients_Translation class
 *
 * @package WP Delicious API Core
 * @since 1.7.8
 */

/**
 * Core class used to implement the Delicious Recipes REST API ingredients translation.
 *
 * @since 1.7.8
 */

class Delicious_Recipes_Pro_REST_Ingredients_Translation {

	/**
	 * The namespace of this controller's route.
	 *
	 * @var string
	 */
	protected $namespace = 'deliciousrecipe/v1';

	/**
	 * Register the routes for the objects of the controller.
	 *
	 * @return void
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/ingredients-translation',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_translated_texts' ),
					'permission_callback' => array( $this, 'get_translated_texts_permissions_check' ),
				),
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'save_translated_texts' ),
					'permission_callback' => array( $this, 'save_translated_texts_permissions_check' ),
				),
			)
		);
	}

	/**
	 * Check if a given request has access to get recipes.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 * @return WP_Error|bool
	 */
	public function get_translated_texts_permissions_check( $request ) {
		if ( ! current_user_can( 'manage_options' ) ) {
			return new \WP_Error(
				'rest_forbidden_context',
				esc_html__( 'You cannot get translated texts.', 'delicious-recipes-pro' ),
				array( 'status' => rest_authorization_required_code() )
			);
		}
		return true;
	}

	/**
	 * Check if a given request has access to get recipes by id.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 * @return WP_Error|bool
	 */
	public function save_translated_texts_permissions_check( $request ) {
		if ( ! current_user_can( 'manage_options' ) ) {
			return new \WP_Error(
				'rest_forbidden_context',
				esc_html__( 'You cannot save translated texts.', 'delicious-recipes-pro' ),
				array( 'status' => rest_authorization_required_code() )
			);
		}
		return true;
	}

	/**
	 * Sets Authorization Status Code.
	 *
	 * @return $status
	 */
	public function rest_authorization_required_code() {
		$status = 401;

		if ( is_user_logged_in() ) {
			$status = 403;
		}

		return $status;
	}

	/**
	 * Get translated texts.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 */
	public function get_translated_texts( $request ) {

		$translations = get_option('wpd_ingredients_translation_dictionary', []);

		$data = array(
			'success' => true,
			'message' => __( 'Translated Ingredients Found.', 'delicious-recipes-pro' ),
			'data'    => $translations,
		);

		return $data;
	}

	/**
	 * Save translated texts.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 */
    public function save_translated_texts($request) {
        $formdata = $request->get_json_params();
    
        // Validate input data
        if (empty($formdata['updated_translation_dictionary']) || !is_array($formdata['updated_translation_dictionary'])) {
            return new WP_Error('invalid_data', __('Invalid input data.', 'delicious-recipes-pro'), array('status' => 400));
        }
    
        $updated_translation_dictionary = $formdata['updated_translation_dictionary'];
        $translations = get_option('wpd_ingredients_translation_dictionary', []);
    
        // Ensure translations is an array
        if (!is_array($translations)) {
            $translations = [];
        }
    
        $translations = array_merge($translations, $updated_translation_dictionary);
        update_option('wpd_ingredients_translation_dictionary', $translations);
    
        return [
            'success' => true,
            'message' => __('Translation dictionary updated successfully.', 'delicious-recipes-pro'),
            'data'    => $translations,
        ];
    }
}

/**
 * Register the Delicious Recipes REST API Search Recipes.
 */
function delicious_recipes_register_rest_ingredients_translation() {
	$controller = new Delicious_Recipes_Pro_REST_Ingredients_Translation();
	$controller->register_routes();
}

add_action( 'rest_api_init', 'delicious_recipes_register_rest_ingredients_translation' );
