<?php
/**
 * REST API: Delicious_Recipes_REST_Recipe_Submission_Controller class
 *
 * @package WP Delicious API Core
 * @since 1.6.1
 */

/**
 * Core class used to implement the Delicious Recipes REST API Recipe Submission Controller.
 *
 * @since 1.6.1
 */
class Delicious_Recipes_Pro_REST_Recipe_Submission_Controller {

	/**
	 * The namespace of this controller's route.
	 *
	 * @var string
	 */
	protected $namespace = 'deliciousrecipe/v1';

	/**
	 * Register the routes for the objects of the controller.
	 *
	 * @return void
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/recipe-recaptcha',
			array(
				array(
					'methods'             => WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'recipe_recaptcha' ),
					'permission_callback' => array( $this, 'recipe_recaptcha_permissions_check' ),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/recipe-submission',
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_recipe_submission' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
				),
				array(
					'methods'             => \WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'post_recipe_submission' ),
					'permission_callback' => array( $this, 'update_settings_permissions_check' ),
				),
			)
		);
	}

	/**
	 * Check permissions for the posts.
	 *
	 * @param WP_REST_Request $request Current request.
	 */
	public function get_items_permissions_check( $request ) {
		if ( ! current_user_can( 'read' ) ) {
			return new WP_Error( 'rest_forbidden', esc_html__( 'You cannot view the post resource.', 'delicious-recipes-pro' ), array( 'status' => $this->authorization_status_code() ) );
		}
		return true;
	}

	/**
	 * Check permissions for the settings update.
	 *
	 * @param WP_REST_Request $request Current request.
	 */
	public function update_settings_permissions_check( $request ) {
		// Check manage_options prevlages.
		if ( ! current_user_can( 'manage_options' ) ) {
			return new WP_Error( 'rest_forbidden', esc_html__( 'You cannot update the site settings.', 'delicious-recipes-pro' ), array( 'status' => $this->authorization_status_code() ) );
		}
		return true;
	}

	/**
	 * Sets Authorization Status Code.
	 *
	 * @return $status
	 */
	public function authorization_status_code() {
		$status = 401;

		if ( is_user_logged_in() ) {
			$status = 403;
		}

		return $status;
	}

	/**
	 * Get the recipe submission settings.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 */
	public function get_recipe_submission( $request ) {
		$global_settings                      = get_option( 'delicious_recipe_settings', array() );
		$recipe_submission_structure          = get_option( 'recipe_submission_structure', array() );
		$recipe_fields_order                  = array(
			array(
				'id'           => 'recipe-details',
				'sectionTitle' => __( 'Recipe Details', 'delicious-recipes-pro' ),
				'enable'       => array( 'yes' ),
			),
			array(
				'id'           => 'recipe-info',
				'sectionTitle' => __( 'Recipe Card Information', 'delicious-recipes-pro' ),
				'enable'       => array( 'yes' ),
			),
			array(
				'id'           => 'recipe-ingredients',
				'sectionTitle' => __( 'Ingredients Information', 'delicious-recipes-pro' ),
				'enable'       => array( 'yes' ),
			),
			array(
				'id'           => 'recipe-instructions',
				'sectionTitle' => __( 'Instructions Information', 'delicious-recipes-pro' ),
				'enable'       => array( 'yes' ),
			),
			array(
				'id'           => 'recipe-gallery',
				'sectionTitle' => __( 'Recipe Gallery', 'delicious-recipes-pro' ),
				'enable'       => array( 'yes' ),
			),
			array(
				'id'           => 'recipe-others',
				'sectionTitle' => __( 'Others', 'delicious-recipes-pro' ),
				'enable'       => array( 'yes' ),
			),
		);
		$global_settings['recipeFieldsOrder'] = $recipe_fields_order;
		update_option( 'delicious_recipe_settings', $global_settings );
		return wp_send_json_success( $recipe_submission_structure );
	}

	/**
	 * Update the recipe submission settings.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 */
	public function post_recipe_submission( $request ) {
		$params = $request->get_json_params();
		if ( ! isset( $params[1] ) && ! wp_verify_nonce( $params[0], 'wp_recipe_submission_admin' ) ) {
			return rest_ensure_response(
				array(
					'success' => false,
					'message' => __( 'Nonce verification failed.', 'delicious-recipes-pro' ),
					'data'    => array(),
				)
			);
		}
		update_option( 'recipe_submission_structure', $params[0] );
		return wp_send_json_success( $params );
	}

	/**
	 * Check if a given request has access to submit a recipe.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 * @return WP_Error|bool
	 */
	public function recipe_recaptcha_permissions_check( $request ) {
		if ( ! current_user_can( 'manage_options' ) ) {
			return new \WP_Error(
				'rest_forbidden_context',
				esc_html__( 'You cannot submit a recipe.', 'delicious-recipes-pro' ),
				array( 'status' => rest_authorization_required_code() )
			);
		}
		return true;
	}

	/**
	 * Submit a recipe.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 * @throws Exception If the Google reCAPTCHA validation fails.
	 */
	public function recipe_recaptcha( $request ) {
		$global_settings          = delicious_recipes_get_global_settings();
		$google_recaptcha_enabled = false;
		if ( isset( $global_settings['recaptchaEnabledForRecipe'] ) && is_array( $global_settings['recaptchaEnabledForRecipe'] ) && isset( $global_settings['recaptchaEnabledForRecipe'][0] ) && 'yes' === $global_settings['recaptchaEnabledForRecipe'][0] ) {
			$google_recaptcha_enabled = true;
		}

		if ( $google_recaptcha_enabled ) {
			$form_response = $request->get_json_params();
			$google_token  = isset( $form_response['googleToken'] ) ? $form_response['googleToken'] : false;

			$google_secret = isset( $global_settings['recaptchaSecretKey'] ) && ! empty( $global_settings['recaptchaSecretKey'] ) ? $global_settings['recaptchaSecretKey'] : false;

			if ( $google_recaptcha_enabled && $google_secret ) {
				$google_response = wp_remote_post(
					'https://www.google.com/recaptcha/api/siteverify',
					array(
						'body'    => array(
							'secret'   => $google_secret,
							'response' => $google_token,
						),
						'headers' => array(),
					)
				);

				if ( 200 !== $google_response['response']['code'] ) {
					wp_send_json_error(
						array(
							'success' => false,
							'message' => __( 'Google reCAPTCHA validation failed.', 'delicious-recipes-pro' ),
						)
					);
				} else {
					wp_send_json_success(
						array(
							'success' => true,
							'message' => __( 'Google reCAPTCHA validation successful.', 'delicious-recipes-pro' ),
						)
					);
				}
			}
		}
		wp_send_json_success(
			array(
				'success' => true,
				'message' => __( 'Google reCAPTCHA validation not required.', 'delicious-recipes-pro' ),
			)
		);
	}
}

/**
 * Register the Recipe Submission REST API routes.
 */
function delicious_recipes_register_recipe_rest_routes() {
	$controller = new Delicious_Recipes_Pro_REST_Recipe_Submission_Controller();
	$controller->register_routes();
}

add_action( 'rest_api_init', 'delicious_recipes_register_recipe_rest_routes' );
