<?php
/**
 * REST API: Delicious_Recipes_Pro_REST_Search_Recipes class
 *
 * @package WP Delicious API Core
 * @since 1.7.8
 */

/**
 * Core class used to implement the Delicious Recipes REST API search Recipes.
 *
 * @since 1.7.8
 */

class Delicious_Recipes_Pro_REST_Search_Recipes {

	/**
	 * The namespace of this controller's route.
	 *
	 * @var string
	 */
	protected $namespace = 'deliciousrecipe/v1';

	/**
	 * Register the routes for the objects of the controller.
	 *
	 * @return void
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/search-recipes',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_recipes_to_search' ),
					'permission_callback' => array( $this, 'get_recipes_to_search_permissions_check' ),
				),
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'get_recipe_by_id' ),
					'permission_callback' => array( $this, 'get_recipe_by_id_permissions_check' ),
				),
			)
		);
	}

	/**
	 * Check if a given request has access to get recipes.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 * @return WP_Error|bool
	 */
	public function get_recipes_to_search_permissions_check( $request ) {
		if ( ! current_user_can( 'manage_options' ) ) {
			return new \WP_Error(
				'rest_forbidden_context',
				esc_html__( 'You cannot get review tags.', 'delicious-recipes-pro' ),
				array( 'status' => rest_authorization_required_code() )
			);
		}
		return true;
	}

	/**
	 * Check if a given request has access to get recipes by id.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 * @return WP_Error|bool
	 */
	public function get_recipe_by_id_permissions_check( $request ) {
		if ( ! current_user_can( 'manage_options' ) ) {
			return new \WP_Error(
				'rest_forbidden_context',
				esc_html__( 'You cannot update review tags.', 'delicious-recipes-pro' ),
				array( 'status' => rest_authorization_required_code() )
			);
		}
		return true;
	}

	/**
	 * Sets Authorization Status Code.
	 *
	 * @return $status
	 */
	public function rest_authorization_required_code() {
		$status = 401;

		if ( is_user_logged_in() ) {
			$status = 403;
		}

		return $status;
	}

	/**
	 * Get All Recipes for searching.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 */
	public function get_recipes_to_search( $request ) {

		$results = get_posts(
			array(
				'post_type'      => DELICIOUS_RECIPE_POST_TYPE,
				'posts_per_page' => -1,
			)
		);
		$recipe  = array();
		foreach ( $results as $result ) {
			$recipe_id    = $result->ID;
			$recipe_title = $result->post_title;
			$recipe[]     = array(
				'ID'                 => $recipe_id,
				'name'               => $recipe_title,
				'recipe_description' => $result->post_content,
				'permalink'          => get_permalink( $recipe_id ),
			);
		}

		$data = array(
			'success' => true,
			'message' => __( 'Review tags found.', 'delicious-recipes-pro' ),
			'data'    => $recipe,
		);
		return $data;
	}

	/**
	 * Get Recipe by ID.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 */
	public function get_recipe_by_id( $request ) {
		$formdata = $request->get_json_params();
		$post_id  = $formdata['id'];
		$recipe   = delicious_recipes_get_recipe( get_post( $post_id ) );

		$icon_array = array();
		if ( is_object( $recipe ) && $recipe->recipe_keys ) {
			foreach ( $recipe->recipe_keys as $recipe_key ) {
				$key              = get_term_by( 'name', $recipe_key, 'recipe-key' );
				$recipe_key_metas = get_term_meta( $key->term_id, 'dr_taxonomy_metas', true );
				$key_svg          = isset( $recipe_key_metas['taxonomy_svg'] ) ? $recipe_key_metas['taxonomy_svg'] : '';
				$icon             = delicious_recipes_get_svg( $key_svg, 'recipe-keys', '#000000' );
				array_push( $icon_array, $icon );
			}
			$recipe->icon = $icon_array;
		}

		$data = array(
			'success' => true,
			'message' => __( 'Review tags updated.', 'delicious-recipes-pro' ),
			'data'    => $recipe,
		);

		return $data;
	}
}

/**
 * Register the Delicious Recipes REST API Search Recipes.
 */
function delicious_recipes_register_recipe_search_routes() {
	$controller = new Delicious_Recipes_Pro_REST_Search_Recipes();
	$controller->register_routes();
}

add_action( 'rest_api_init', 'delicious_recipes_register_recipe_search_routes' );
